clear all;
close all;
clc;
%% Step 1: Merge CH4 & MET data
% Read the CH4 and MET data
ch4 = readtable("\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Eddy Covariance\8_Hz\CH4_MGGA.csv");
met = readtable("\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Eddy Covariance\8_Hz\MET.csv");

% Ensure Time columns are datetime format
ch4.Time = datetime(ch4.Time, 'InputFormat', 'MM/dd/yyyy HH:mm:ss.SSS');
met.Time = datetime(met.Time, 'InputFormat', 'MM/dd/yyyy HH:mm:ss.SSS');

% Fix column names in CH4 and MET tables
ch4.Properties.VariableNames = matlab.lang.makeValidName(ch4.Properties.VariableNames);
met.Properties.VariableNames = matlab.lang.makeValidName(met.Properties.VariableNames);

% Resample CH4 data to 8 Hz using linear interpolation
% Create new timestamps at 8 Hz (every 125 milliseconds)
min_time = min(ch4.Time);
max_time = max(ch4.Time);
new_times = (min_time:seconds(0.125):max_time)';

% Interpolate CH4 data
columns_to_interpolate = {'x_CH4__ppm', 'x_CH4_d_ppm', 'GasP_torr', 'GasT_C', 'Frequency_CH4'};
interpolated_data = table(new_times, 'VariableNames', {'Time'});
for col = columns_to_interpolate
    col_name = matlab.lang.makeValidName(col{1}); % Ensure column name is valid
    interpolated_data.(col_name) = interp1(ch4.Time, ch4.(col_name), new_times, 'linear', 'extrap');
end

% Interpolate MET data into the interpolated_data table
met_columns_to_interpolate = {'AT', 'BP', 'RH', 'U', 'V', 'W', 'WS', 'WD', 'Frequency'};
for col = met_columns_to_interpolate
    col_name = matlab.lang.makeValidName(col{1}); % Ensure column name is valid
    interpolated_data.(col_name) = interp1(met.Time, met.(col_name), new_times, 'linear', 'extrap');
end


% Save the table as a CSV file
output_path = "\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Eddy Covariance\8_Hz\";
csv_file = fullfile(output_path, 'MET_CH4_8hz.csv');
writetable(interpolated_data, csv_file);

disp('Step 1 complete');


%% Step 2: Merge met_ch4 to releases
% Read releases table
interpolated_data = readtable("\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Eddy Covariance\8_Hz\MET_CH4_8hz.csv");
releases = readtable('\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Releases\METEC_Emissions.csv');

% Convert UTC start and end times to Mountain Time (America/Denver)
utcTimeZone = 'UTC';
mtTimeZone = 'America/Denver';

% Convert UTC start and end times to datetime objects
releases.UTCStart = datetime(releases.UTCStart, 'InputFormat', 'yyyy-MM-dd HH:mm:ss', 'TimeZone', utcTimeZone);
releases.UTCEnd = datetime(releases.UTCEnd, 'InputFormat', 'yyyy-MM-dd HH:mm:ss', 'TimeZone', utcTimeZone);

% Convert UTC start and end times to Mountain Time
releases.MTStart = datetime(releases.UTCStart, 'TimeZone', mtTimeZone);
releases.MTEnd = datetime(releases.UTCEnd, 'TimeZone', mtTimeZone);

                
                % Initialize a new empty table to store the merged data
                mergedTable = table();
                               
                % Iterate over each row in releases
                for i = 1:height(releases)
                    % Extract the MTStart and MTEnd from releases for the current row
                    MTStart = releases.MTStart(i);
                    MTEnd = releases.MTEnd(i);

                    % Convert interpolated_data.Time to the "America/Denver" timezone
                    interpolated_data.Time = datetime(interpolated_data.Time, 'TimeZone', 'America/Denver');

                    
                    % Find rows in met_ch4 where the Time falls within the MTStart and MTEnd interval
                    mask = interpolated_data.Time >= MTStart & interpolated_data.Time <= MTEnd;
                    metInRange = interpolated_data(mask, :);
                    
                    % If there are rows in metInRange, add the release information from releases
                    if ~isempty(metInRange)
                        % Add the release information from releases to metInRange
                        % by adding new columns for each variable in releases
                        releaseData = repmat(table2struct(releases(i, :)), height(metInRange), 1);
                        releaseData = struct2table(releaseData, 'AsArray', true); % Specify 'AsArray' parameter
                        metInRange = [metInRange, releaseData];
                        
                        % Append the merged data to mergedTable
                        mergedTable = [mergedTable; metInRange];
                    end
                end
   disp('Step 2 complete');

%% Step 3: Include mast (MGGA) location to the table

                MGGA_lat = repmat(40.59603590, height(mergedTable), 1); 
                MGGA_long = repmat(-105.14032186, height(mergedTable), 1);
                mergedTable = addvars(mergedTable, MGGA_lat, MGGA_long, 'NewVariableNames', {'MGGA_lat_N', 'MGGA_long_W'});
               
 disp('Step 3 complete')


 %% % Step 4: Split MGGA_met_releases mergedTable into event tables 

% Convert datetime values to serial date numbers
MTStart_serial = datenum(mergedTable.MTStart);
MTEnd_serial = datenum(mergedTable.MTEnd);

% Round off the serial date numbers to the nearest minute
rounded_MTStart = round(MTStart_serial * 24 * 60) / (24 * 60);
rounded_MTEnd = round(MTEnd_serial * 24 * 60) / (24 * 60);

% Find unique combinations of rounded MTStart and MTEnd
unique_combinations = unique([rounded_MTStart, rounded_MTEnd], 'rows');

% Initialize a cell array to store mini tables
mini_tables = cell(size(unique_combinations, 1), 1);

% Loop through each unique combination and create mini tables
for i = 1:size(unique_combinations, 1)
    % Find rows that match the current combination
    matching_rows = all([rounded_MTStart, rounded_MTEnd] == unique_combinations(i, :), 2);
    
    % Create a mini table for the current combination
    mini_tables{i} = mergedTable(matching_rows, :);
end

% Define the directory where the event tables will be saved
save_directory = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Eddy Covariance\8_Hz\Event Tables';

% Check if the directory exists, if not, create it
if ~exist(save_directory, 'dir')
    mkdir(save_directory);
end

% Loop through each mini table and save it as an event table
for i = 1:numel(mini_tables)
    % Get the start and end times for the current mini table
    MTStart = mini_tables{i}.MTStart(1);
    MTEnd = mini_tables{i}.MTEnd(1);
    
    % Create the filename for the event table
    filename = sprintf('%s_%s Event table.csv', datestr(MTStart, 'yyyymmddHHMM'), datestr(MTEnd, 'yyyymmddHHMM'));
    
    % Save the mini table as a CSV file in the specified directory
    writetable(mini_tables{i}, fullfile(save_directory, filename));
end

disp('Step 4 complete');

%% %Step 5: Add distance to the tables

% Define the haversine function
function distance = haversine(lat1, lon1, lat2, lon2)
    % Earth radius in meters
    R = 6371000;

    % Convert degrees to radians
    lat1 = deg2rad(lat1);
    lon1 = deg2rad(lon1);
    lat2 = deg2rad(lat2);
    lon2 = deg2rad(lon2);

    % Calculate differences
    dlat = lat2 - lat1;
    dlon = lon2 - lon1;

    % Haversine formula
    a = sin(dlat/2).^2 + cos(lat1) .* cos(lat2) .* sin(dlon/2).^2;
    c = 2 * atan2(sqrt(a), sqrt(1-a));
    distance = R * c;
end

% Define the directory path
path = "\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Eddy Covariance\8_Hz\Event Tables";

         % Get a list of all CSV files 
        file_list = dir(fullfile(path, '*.csv'));

        % Loop through each CSV file 
        for j = 1:numel(file_list)
            % Read the table from the CSV file
            filename = fullfile(path, file_list(j).name);
            file = readtable(filename);

            % Calculate distances
            distance_meters = haversine(file.Lat, file.Lon, file.MGGA_lat_N, file.MGGA_long_W);

            % Add distances to the file table
            file.Distance_meters = distance_meters;

            % Write the modified table back to the original CSV file
            writetable(file, filename);
        end
    disp('Step 5 Complete')
    %% 
